<?php
/**
 * Sets up custom filters and actions for the theme.  This does things like sets up sidebars, menus, scripts,
 * and lots of other awesome stuff that WordPress themes do.
 */

/* Register custom image sizes. */
add_action( 'init', 'intruder_register_image_sizes', 5 );

/* Register custom menus. */
add_action( 'init', 'intruder_register_menus', 5 );

/* Register sidebars. */
add_action( 'widgets_init', 'intruder_register_sidebars', 5 );

/* Register theme layouts. */
add_action( 'hybrid_register_layouts', 'intruder_register_layouts' );

/* Add custom scripts. */
add_action( 'wp_enqueue_scripts', 'intruder_enqueue_scripts' );

/* Register custom styles. */
add_action( 'wp_enqueue_scripts', 'intruder_enqueue_styles', 0 );

/* Filters the excerpt length. */
add_filter( 'excerpt_length', 'intruder_excerpt_length' );

/* Remove read more in excerpt */
add_filter( 'excerpt_more', 'intruder_read_more' );

/* Main theme layout */
add_filter( 'theme_mod_theme_layout', 'intruder_main_layout' );

/* Adds custom attributes to the subsidiary sidebar. */
add_filter( 'hybrid_attr_sidebar', 'intruder_sidebar_subsidiary_class', 10, 2 );

/* Remove allowed tags from comment form */
add_filter( 'comment_form_defaults', 'intruder_cf_allowed_tags' );

/* Site logo. */
add_filter( 'hybrid_site_title', 'intruder_logo' );

/* Add class to image link. */
add_filter( 'get_the_image', 'intruder_add_img_class' );

/* Adds placeholders to form elements. */
add_filter( 'comment_form_defaults', 'intruder_comment_form' );

/* Register theme widgets. */
add_action( 'widgets_init', 'intruder_register_widgets' );

/* Sets custom archive title. */
add_filter( 'get_the_archive_title', 'intruder_archive_title' );

/* Sets tag number to 7 in widget tags. */
add_filter( 'widget_tag_cloud_args', 'intruder_set_tag_number' );

/* Add extra body classes. */
add_filter( 'body_class', 'intruder_body_classes' );

/**
 * Registers custom image sizes for the theme.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function intruder_register_image_sizes() {

	/* Sets the 'post-thumbnail' size. */
	set_post_thumbnail_size( 150, 150, true );

	/* Widget image. */
	add_image_size( 'intruder-widget-image', 85, 85, true );

	/* Front page image size. */
	add_image_size( 'intruder-full', 1140, 550, true );
}

/**
 * Registers nav menu locations.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function intruder_register_menus() {
	register_nav_menu( 'primary',   _x( 'Primary', 	'nav menu location', 'intruder' ) );
	register_nav_menu( 'social',    _x( 'Social', 	'nav menu location', 'intruder' ) );
}

/**
 * Registers sidebars.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function intruder_register_sidebars() {

	hybrid_register_sidebar(
		array(
			'id'          => 'primary',
			'name'        => _x( 'Primary', 'sidebar', 'intruder' ),
			'description' => __( 'The main sidebar. It is displayed on either the left or right side of the page based on the chosen layout.', 'intruder' )
		)
	);

	hybrid_register_sidebar(
		array(
			'id'          => 'subsidiary',
			'name'        => _x( 'Subsidiary', 'sidebar', 'intruder' ),
			'description' => __( 'A sidebar located in the footer of the site. Optimized for one, two, three or four widgets (and multiples thereof).', 'intruder' )
		)
	);

}

/**
 * Adds support for multiple theme layouts.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function intruder_register_layouts() {

    hybrid_register_layout(
        '2c-l',
        array(
            'label'              => __( '2 Columns: Content / Sidebar', 'intruder' ),
            'show_in_customizer' => true,
            'show_in_meta_box'   => true,
            'image'              => THEME_URI . 'images/admin/2c-l.png'
        )
    );

    hybrid_register_layout(
        '2c-r',
        array(
            'label'              => __( '2 Columns: Sidebar / Content', 'intruder' ),
            'show_in_customizer' => true,
            'show_in_meta_box'   => true,
         	'image'              => THEME_URI . 'images/admin/2c-r.png'
        )
    );

    hybrid_register_layout(
        '1c',
        array(
            'label'              => __( '1 Column', 'intruder' ),
            'show_in_customizer' => true,
            'show_in_meta_box'   => true,
            'image'              => THEME_URI . 'images/admin/1c.png'
        )
    );

}

/**
 * Enqueues scripts.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function intruder_enqueue_scripts() {

	/* Minified or dev css / js versions. */
	$suffix = hybrid_get_min_suffix();

	/* Register theme js setup file. */
	wp_register_script(
		'intruder-setup-js',
		THEME_URI . "js/setup{$suffix}.js",
		array( 'jquery' ),
		null,
		true
	);

	/* Register fitvids script for responsive videos. */
	wp_register_script(
		'intruder-fitvids',
		THEME_URI . "js/jquery.fitvids{$suffix}.js",
		array( 'jquery' ),
		null,
		true
	);

	/* Register bxslider script for sliders, carousels, tickers... */
	wp_register_script(
		'intruder-slider',
		THEME_URI . "js/jquery.bxslider{$suffix}.js",
		array( 'jquery' ),
		null,
		true
	);

	/* Register circular progress script. */
	wp_register_script(
		'intruder-circle-progress',
		THEME_URI . "js/circle-progress{$suffix}.js",
		array( 'jquery' ),
		null,
		true
	);

	wp_register_script(
		'intruder-inview',
		THEME_URI . "js/jquery.inview{$suffix}.js",
		array( 'jquery' ),
		null,
		true
	);


	/*================================
	=            ENQUEUE!            =
	================================*/

	/* Check if Live Composer is active, and load all scripts for modules. */
	if ( intruder_is_active_lc() ) {
		wp_enqueue_script( 'intruder-slider' );
		wp_enqueue_script( 'intruder-inview' );
		wp_enqueue_script( 'intruder-circle-progress' );
	}

	/* Load the rest. */
	wp_enqueue_script( 'intruder-fitvids' );
	wp_enqueue_script( 'intruder-setup-js' );

}

/**
 * Registers custom stylesheets for the front end.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function intruder_enqueue_styles() {

	$suffix = hybrid_get_min_suffix();
	$style  = is_rtl() ? 'css/rtl' : 'style';

	/* Load Merriweather font. */
	wp_register_style( 'merriweather', '//fonts.googleapis.com/css?family=Merriweather:400,700,700italic|Merriweather+Sans:300,300italic,400,400italic,700,700italic&subset=latin,latin-ext' );
	wp_enqueue_style( 'merriweather' );


	/* Font Icon Font */
	wp_register_style( 'font-awesome', THEME_URI . "css/font-awesome{$suffix}.css" );
	wp_enqueue_style( 'font-awesome' );

	/* Cleaner gallery styles. */
	wp_enqueue_style( 'hybrid-gallery' );

	$style_url = THEME_URI . "{$style}{$suffix}.css";

	if ( ! file_exists( $style_url ) )
		$style_url = THEME_URI . "{$style}.css";

	/* Load parent theme stylesheet. */
	wp_register_style( 'style', $style_url );
	wp_enqueue_style ( 'style' );

	/* Autoload child theme stylesheet. */
	if ( is_child_theme() )
		wp_enqueue_style( 'child', get_stylesheet_uri() );
}


/**
 * Adds a custom excerpt length.
 *
 * @since  1.0.0
 * @access public
 * @param  int     $length
 * @return int
 */
function intruder_excerpt_length( $length ) {
	return 45;
}

/**
 * Disables read more link ([...]) in excerpt.
 *
 * @since 1.0.0
 * @access public
 * @param  string 	$more
 * @return string
 */
function intruder_read_more( $more ) {
	return '...';
}

/**
 * Sets default layout for theme and individual post / page.
 *
 * @since  1.0.0
 * @access public
 * @param  string 	$layout
 * @return string   $layout
 */
function intruder_main_layout( $layout ) {

	if ( is_attachment() && wp_attachment_is_image() ) {
		$post_layout = hybrid_get_post_layout( get_queried_object_id() );

		if ( ! $post_layout || 'default' === $post_layout )
			$layout = '1c';
	}

	return $layout;
}

/**
 * Adds a custom class to the 'subsidiary' sidebar.  This is used to determine the number of columns used to
 * display the sidebar's widgets.  This optimizes for 1, 2, 3 and 4 columns or multiples of those values.
 *
 * Note that we're using the global $sidebars_widgets variable here. This is because core has marked
 * wp_get_sidebars_widgets() as a private function. Therefore, this leaves us with $sidebars_widgets for
 * figuring out the widget count.
 * @link http://codex.wordpress.org/Function_Reference/wp_get_sidebars_widgets
 *
 * @since  1.0.0
 * @access public
 * @param  array  $attr
 * @param  string $context
 * @return array
 */
function intruder_sidebar_subsidiary_class( $attr, $context ) {

	if ( 'subsidiary' === $context ) {

		global $sidebars_widgets;

		if ( is_array( $sidebars_widgets ) && !empty( $sidebars_widgets[ $context ] ) ) {

			$count = count( $sidebars_widgets[ $context ] );

			if ( ( $count == 4 ) || $count > 4 )
				$attr['class'] .= ' sidebar-cols-4';

			elseif ( !( $count % 3 ) )
				$attr['class'] .= ' sidebar-cols-3';

			elseif ( !( $count % 2 ) )
				$attr['class'] .= ' sidebar-cols-2';

			else
				$attr['class'] .= ' sidebar-cols-1';

		}
	}

	return $attr;
}


/**
 * Disables comments allowed tags below comment textarea.
 *
 * @since 1.0.0
 * @access public
 * @param  array $defaults
 * @return array
 */
function intruder_cf_allowed_tags( $defaults ) {
	$defaults['comment_notes_after'] = '';
	return $defaults;
}

/**
 * Set's logo, replaces site title.
 * @since 1.0.0
 * @access public
 * @param  string 	$title
 * @return string   $title
 */
function intruder_logo( $title ) {

	$logo       = get_theme_mod( 'logo' );
	$site_title = get_bloginfo( 'name' );

	if ( ! empty( $logo ) ) {
		$img   = '<img src="'. esc_url( $logo ) .'" alt="'. esc_attr( $site_title ) .'" />';
		$title = sprintf( '<h1 id="site-title" class="site-title"><a href="%s" rel="home">%s</a></h1>', home_url(), $img );
	} elseif ( strlen( $site_title ) > 1 && str_word_count( $site_title ) > 1 ) {
		$site_title = explode( ' ', $site_title, 2 );
		$title      = sprintf( '<h1 id="site-title" class="site-title"><a href="%s" rel="home">%s%s %s%s</a></h1>', home_url(), '<span>', esc_html( $site_title[0] ), '</span>', esc_html( $site_title[1] ) );
	}

	return $title;
}

/**
 * Adds .intruder-thumbnail class to image link.
 *
 * @since 1.0.0
 * @access public
 * @param  string 	$image
 * @return string
 */
function intruder_add_img_class( $image ) {
	$image = str_replace( '<a', '<a class="intruder-thumbnail" ', $image );
	return $image;
}

/**
 * Function adds placeholders to inputs.
 *
 * @since 1.0.0
 * @access public
 * @param  array $defaults
 * @return array
 */
function intruder_comment_form( $defaults ) {

	$commenter = wp_get_current_commenter();
	$req       = get_option( 'require_name_email' );
	$aria_req  = ( $req ? " aria-required='true'" : '' );

	$defaults['fields'] = array(

		'author' =>
			'<p class="comment-form-author">' .
			'<input id="author" placeholder="'. __( 'Name', 'luminous' ) . ( $req == 1 ? '*' : '' ) . '" name="author" type="text" value="' . esc_attr( $commenter['comment_author'] ) .
			'" size="30"' . $aria_req . ' /></p>',

		'email' =>
			'<p class="comment-form-email">' .
			'<input id="email" placeholder="'. __( 'Email', 'luminous' ) . ( $req == 1 ? '*' : '' ) . '" name="email" type="text" value="' . esc_attr(  $commenter['comment_author_email'] ) .
			'" size="10"' . $aria_req . ' /></p>',

		'url' =>
			'<p class="comment-form-url">' .
			'<input id="url" name="url" placeholder="'. __( 'Website', 'luminous' ) . '" type="text" value="' . esc_attr( $commenter['comment_author_url'] ) .
			'" size="30" /></p>'

	);

	$defaults['comment_field'] = '<p class="comment-form-comment">'.
								 '<textarea placeholder="'. __( 'Your message', 'luminous' ) .'*" id="comment" name="comment" rows="8" cols="45"'.
								 'aria-required="true"></textarea></p>';

	$defaults['comment_notes_after'] = '';

	return $defaults;
}

/**
 * Registers custom theme widgets.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function intruder_register_widgets() {

	require_once( THEME_DIR . 'inc/widget-popular-posts.php' );
	require_once( THEME_DIR . 'inc/widget-our-info.php' );
	require_once( THEME_DIR . 'inc/widget-company-info.php' );

	register_widget( 'Intruder_Widget_Popular' );
	register_widget( 'Intruder_Our_Info' );
	register_widget( 'Intruder_Widget_Company_Info' );

}

/**
 * Sets custom title from customizer.
 *
 * @since  1.0.0
 * @access public
 * @param  string $title [description]
 * @return string
 */

function intruder_archive_title( $title ) {

	$custom_title = esc_html( get_theme_mod( 'loop_title', __( 'Tips & Tricks' ) ) );

	if ( ! empty( $custom_title ) ) {
		if ( is_front_page() || is_home() || is_singular() )
			return $custom_title;
	}

	return $title;

}

/**
 * Sets number of tags in widgets to 7.
 *
 * @since  1.0.0
 * @access public
 * @param  array $args
 * @return array
 */
function intruder_set_tag_number( $args ) {
	$args['number'] = 7;
	return $args;
}

/**
 * Adds extra body classes, like ".transparent-header".
 *
 * @param  array $classes
 * @return array
 */
function intruder_body_classes( $classes ) {

	if ( intruder_transparent_header() )
		$classes[] = 'transparent-header';

	return $classes;
}

/**
 * Checks whether to show primary sidebar. Sidebar will be shown unless it's 1 column layout.
 *
 * @since  1.0.0
 * @access public
 * @return boolean
 */
function intruder_show_sidebar() {

	$global_layout = esc_html( get_theme_mod( 'theme_layout', '2c-l' ) );

	if ( is_singular() ) {

		if ( is_intruder_front_page() )
			return false;

		$singular_layout = hybrid_get_post_layout( get_the_ID() );

		if ( '1c' === $singular_layout )
			return false;
		elseif ( '' === $singular_layout && $global_layout == '1c' )
			return false;
		else
			return true;
	}
	else return '1c' === $global_layout ? false : true;
}

/**
 * Check whether it's Intruder's front page.
 *
 * @return boolean
 */
function is_intruder_front_page() {
	return is_page_template( 'template-front-page.php' );
}

/**
 * Checks if transparent header is on.
 *
 * @return boolean
 */
function intruder_transparent_header() {
	$th = (bool) get_theme_mod( 'transparent_header', 1 );
	return is_intruder_front_page() && $th;
}

/**
 * Check if Live Composer is installed.
 *
 * @return boolean
 */
function intruder_is_lc_installed() {
	return defined( 'DS_LIVE_COMPOSER_BASENAME' ) ? true : false;
}

/**
 * Check if Live Composer is active.
 *
 * @return [type] [description]
 */
function intruder_is_active_lc() {
	if ( intruder_is_lc_installed() )
		return DS_LIVE_COMPOSER_ACTIVE && is_user_logged_in() && current_user_can( DS_LIVE_COMPOSER_CAPABILITY );
	else
		return false;
}