<?php
/**
 * Sets up custom filters and actions for the theme.  This does things like sets up sidebars, menus, scripts, 
 * and lots of other awesome stuff that WordPress themes do.
 */

/* Register custom image sizes. */
add_action( 'init', 'losatom_register_image_sizes', 5 );

/* Register custom menus. */
add_action( 'init', 'losatom_register_menus', 5 );

/* Register sidebars. */
add_action( 'widgets_init', 'losatom_register_sidebars', 5 );

/* Add custom scripts. */
add_action( 'wp_enqueue_scripts', 'losatom_enqueue_scripts' );

/* Register custom styles. */
add_action( 'wp_enqueue_scripts', 'losatom_register_styles', 0 );

/* Filters the excerpt length. */
add_filter( 'excerpt_length', 'losatom_excerpt_length' );

/* Remove read more in excerpt */
add_filter( 'excerpt_more', 'losatom_read_more' );

/* Main theme layout */
add_filter( 'get_theme_layout', 'main_theme_layout' );

/* Remove allowed tags from comment form */
add_filter( 'comment_form_defaults', 'remove_comment_form_allowed_tags' );

/* Remove gallery inline styles */
add_filter( 'use_default_gallery_style', '__return_false' );

/* Site logo. */
add_filter( 'hybrid_site_title', 'losatom_logo' );

/* Site tagline. */
add_filter( 'hybrid_site_description', 'losatom_site_tagline_toggle' );

/* Replace header class. */
add_filter( 'hybrid_attr_header', 'header_class' );

/* Add class to image link. */
add_filter( 'get_the_image', 'losatom_add_img_class' );
 
/* Adds class to body. */
add_filter( 'body_class', 'losatom_body_classes' );

/* Fixes missing wp title. */
add_filter( 'wp_title', 'losatom_fix_wp_title', 10, 2 );

add_filter( 'hybrid_attr_sidebar', 'csssls', 10, 2 );

function csssls( $attr, $context ) {

	if ( 'subsidiary' === $context ) {
		global $sidebars_widgets;

		if ( is_array( $sidebars_widgets ) && !empty( $sidebars_widgets[ $context ] ) ) 
			$attr['class'] = ' section';
				
	}

	return $attr;

}

/**
 * Registers custom image sizes for the theme.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function losatom_register_image_sizes() {

	/* Sets the 'post-thumbnail' size. */
	set_post_thumbnail_size( 150, 150, true );

	/* Widget image. */
	add_image_size( 'losatom-widget-image', 85, 85, true );        

	/* Front page image size. */
	add_image_size( 'losatom-full', 1140, 550, true );             
	
	/* Latest posts image sizes. */
	add_image_size( 'losatom-lp-2-widget-image', 555, 300, true );       
	add_image_size( 'losatom-lp-3-widget-image', 360, 270, true );       
	add_image_size( 'losatom-lp-4-widget-image', 260, 200, true );       

	/* Testimonial image size. */
	add_image_size( 'losatom-testimonial', 100, 100, true );             
}

/**
 * Registers nav menu locations.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function losatom_register_menus() {
	register_nav_menu( 'subsidiary',    _x( 'Subsidiary',    'nav menu location', 'losatom' ) );
}

/**
 * Registers sidebars.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function losatom_register_sidebars() {

	hybrid_register_sidebar(
		array(
			'id'          => 'primary',
			'name'        => _x( 'Blog: Primary', 'sidebar', 'losatom' ),
			'description' => __( 'The main sidebar. It is displayed on either the left or right side of the page based on the chosen layout.', 'losatom' )
		)
	);			

	hybrid_register_sidebar(
		array(
			'id'          => 'subsidiary',
			'name'        => _x( 'Subsidiary (sitewide)', 'sidebar', 'losatom' ),
			'description' => __( 'A sidebar located in the footer of the site. Shown sitewide, by default displays contact and social widgets.', 'losatom' )
		)
	);	
	
	register_sidebar(
		array(
			'id'          => 'front_page_white',
			'name'        => _x( 'Front Page: White', 'sidebar', 'losatom' ),
			'description' => __( 'The first sidebar / section on front page. Used to display all the other widgets like slider, countdown, video, features, call to action, latest posts,  pricing tables, testimonials...', 'losatom' ),
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widget-title">',
			'after_title'   => '</h3>'						
		)
	);	

}

/**
 * Enqueues scripts.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function losatom_enqueue_scripts() {

	$theme_dir = trailingslashit( get_template_directory_uri() );
	$suffix    = hybrid_get_min_suffix();

	wp_register_script( 
		'losatom-custom-js', 
		$theme_dir . "js/custom{$suffix}.js", 
		array( 'jquery' ), 
		null, 
		true 
	);

	wp_register_script( 
		'losatom-fitvids', 
		$theme_dir . 'js/fitvids.min.js', 
		array( 'jquery' ), 
		null, 
		true 
	);
	
	wp_register_script( 
		'losatom-slider', 
		$theme_dir . 'js/slider.min.js', 
		array( 'jquery' ), 
		null, 
		true 
	);

	wp_register_script( 
		'losatom-countdown', 
		$theme_dir . "js/countdown{$suffix}.js", 
		array( 'jquery' ), 
		null, 
		true 
	);

	$display_on_front = is_losatom_landing_page();


	/* Check if it's front page with sections and load scroll reveal if it is. */
	if ( $display_on_front && get_theme_mod( 'show_animations', 1 ) ) {
		wp_register_script( 
			'losatom-scrollReveal', 
			$theme_dir . 'js/scrollReveal.min.js', 
			array( 'jquery' ), 
			null, 
			true 
		);

		wp_enqueue_script( 'losatom-scrollReveal' );
		wp_localize_script( 'losatom-custom-js', 'animations', array( 'animations' => 'on' ) );					
	}

	/* Checks whether to initialize preloader in custom.js. */
	if ( losatom_show_preloader() ) {
		wp_localize_script( 'losatom-custom-js', 'preloader', array( 'preloader' => 'on' ) );			
	}

	wp_enqueue_script( 'losatom-fitvids' );
	wp_enqueue_script( 'losatom-custom-js' );	
			
}

/**
 * Registers custom stylesheets for the front end.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function losatom_register_styles() {
	
	/* Font Icon Font */
	wp_register_style( 'font-awesome', trailingslashit( get_template_directory_uri() ) . 'admin/css/font-awesome.min.css' );
	wp_enqueue_style( 'font-awesome' );

}
 

/**
 * Adds a custom excerpt length.
 *
 * @since  1.0.0
 * @access public
 * @param  int     $length
 * @return int
 */
function losatom_excerpt_length( $length ) {
	return 60;
}

/**
 * Disables read more link ([...]) in excerpt.
 * @since 1.0.0
 * @access public
 * @param  string 	$more 
 * @return string     
 */
function losatom_read_more( $more ) {
	return '...';	
}

/**
 * Sets default layout for theme and for individual post / page.
 *
 * @since  1.0.0
 * @access public
 * @param  string 	$layout 
 * @return string
 */
function main_theme_layout( $layout ) {
	
	$layout          = get_theme_mod( 'theme_layout', '2c-l' );
	$singular_layout = get_post_layout( get_the_ID() );

	if ( is_singular() && $singular_layout != 'default' )
		$layout = $singular_layout;	

	$front = get_option( 'show_on_front' );

	if ( is_attachment() && wp_attachment_is_image() || is_front_page() && ( $front === 'posts' ) ) {
 
		if ( 'default' === $singular_layout )
			$layout = '1c';
	}

	
	return 'layout-' . $layout;
}


/**
 * Disables comments allowed tags below comment textarea.
 * 
 * @since 1.0.0
 * @access public
 * @param  array $defaults 
 * @return array           
 */
function remove_comment_form_allowed_tags( $defaults ) {
	$defaults['comment_notes_after'] = '';
	return $defaults;
}

/**
 * Set's logo, replaces site title. 
 * @since 1.0.0
 * @access public
 * @param  string 	$title 
 * @return string
 */
function losatom_logo( $title ) {

	$logo = get_theme_mod( 'logo', trailingslashit( get_template_directory_uri() ) . 'images/logo.png' );
	
	if ( ! empty( $logo ) ) {
		$alt   = get_bloginfo( 'name' );
		$img   = '<img src="'. esc_url( $logo ) .'" alt="'. esc_attr( $alt ) .'" />';
		$title = sprintf( '<h1 id="site-title"><a href="%s" rel="home">%s</a></h1>', home_url(), $img );
	}

	return $title;
}

/**
 * Shows / hides site tagline.
 * 
 * @since 1.0.0 
 * @access public
 * @param  string 	$tagline 
 * @return string|bool
 */
function losatom_site_tagline_toggle( $tagline ) {
	$state = get_theme_mod( 'site_tagline_toggle', false );
	return ( 1 === $state ) ? $tagline : false;
}

/**
 * Modifies hybrid_attr_header, adds class .wrap to header tag.
 *
 * @since  1.0.0
 * @access public
 * @param array $attr 
 * @return array
 */
function header_class( $attr ) {
	$attr['class'] = 'wrap';
	return $attr;
}

/**
 * Adds .losatom-thumbnail class to image link.
 * 
 * @since 1.0.0
 * @access public 
 * @param  string 	$image
 * @return string
 */
function losatom_add_img_class( $image ) {
	$image = str_replace( '<a', '<a class="losatom-thumbnail" ', $image );
	return $image;
}

/**
 * If theme is set to show posts on front page, add class .full-width-fp to stretch #main to 100%, 
 * otherwise keep it 1140px and show loop meta data. Look misc/loop-meta.php.
 * 
 * @since 1.0.0
 * @access public
 * @param  array 	$classes 
 * @return array
 */
function losatom_body_classes( $classes ) {

	if ( is_front_page() && 'posts' === get_option( 'show_on_front' ) ) {
		$classes[] = 'full-width-fp';
	} else {
		$classes[] = 'show-loop-meta';
	}

	return $classes;
}

/**
 * Checks if class show-loop-meta exists in body class, and if it does returns true. 
 * This way theme checks whether to display loop-meta.php and subsidiary.php sidebar.
 * 
 * @since 1.0.0
 * @access public
 * @return bool
 */
function losatom_display_on_front() {
	return in_array( 'show-loop-meta', get_body_class() );
}

/**
 * Checks if visitor is on Losatom's landing (default) page.
 * 
 * @since 1.0.0
 * @access public
 * @return boolean 
 */
function is_losatom_landing_page() {
	return ! losatom_display_on_front();
}

/**
 * Fixes missing wp title when static page is selected to be on front page, 
 * and nothing is selected in front page drop down (0 selected).
 *
 * @since 1.0.0
 * @access public
 * @param  string 	$title
 * @return string
 */
function losatom_fix_wp_title( $title ) {
	if ( is_home() && get_option( 'page_on_front' ) == 0 )	
		return get_bloginfo( 'name' );
	else
		return $title;
}

/**
 * Checks whether theme should display preloader on front page.
 * 
 * @since 1.0.0
 * @access public
 * @return bool
 */
function losatom_show_preloader() {	
	return is_losatom_landing_page() && get_theme_mod( 'show_preloader', 1 );
}

/**
 * Backward compatibility for title tag.
 * 
 * @since 1.0.0 
 * @access public
 */
if ( ! function_exists( '_wp_render_title_tag' ) ) {
	function losatom_render_title_tag() {
?>
<title><?php wp_title( ':', true, 'right' ); ?></title>
<?php
	}
	add_action( 'wp_head', 'losatom_render_title_tag' );
}