<?php
/**
 * Sets up custom filters and actions for the theme.  This does things like sets up sidebars, menus, scripts, 
 * and lots of other awesome stuff that WordPress themes do.
 */

/* Register custom image sizes. */
add_action( 'init', 'swt_register_image_sizes', 5 );

/* Register custom menus. */
add_action( 'init', 'swt_register_menus', 5 );

/* Register sidebars. */
add_action( 'widgets_init', 'swt_register_sidebars', 5 );

/* Add custom scripts. */
add_action( 'wp_enqueue_scripts', 'swt_enqueue_scripts' );

/* Register custom styles. */
add_action( 'wp_enqueue_scripts',    'swt_register_styles', 0 );

/* Filters the excerpt length. */
add_filter( 'excerpt_length', 'swt_excerpt_length' );

/* Remove read more in excerpt */
add_filter( 'excerpt_more', 'swt_read_more' );

/* Main theme layout */
add_filter( 'get_theme_layout', 'main_theme_layout' );

/* Add column number class to sidebar */
add_filter( 'hybrid_attr_sidebar', 'footer_column_number', 10, 2 );
add_filter( 'hybrid_attr_loop-meta', 'loop_meta_class', 10, 2 );

/* Disable sidebar if 1 column layout */
add_filter( 'sidebars_widgets', 'swt_disable_sidebars' );
add_action( 'template_redirect', 'swt_one_column' );	

/* Modifies default comment form */
add_filter( 'comment_form_defaults', 'swt_modify_commentform' );

/* Add header scripts, custom css and favicon to <head> */
add_action( 'wp_head', 'swt_head_hooks' );

/* Add footer scripts to </body> */
add_action( 'wp_footer', 'swt_footer_scripts' ); 

/* Adds span around site title letters */
add_filter( 'hybrid_site_title', 'add_span_2_site_title' );

/* Adds extra classes to body */
add_filter( 'body_class', 'extra_body_classes' );

/* Remove default gallery inline styles */
add_filter( 'use_default_gallery_style', '__return_false' );

/* Remove first gallery from post content for gallery post format */
add_filter( 'the_content', 'strip_first_gallery' );

/**
 * Registers custom image sizes for the theme.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function swt_register_image_sizes() {

	/* Sets the 'post-thumbnail' size. */
	set_post_thumbnail_size( 150, 150, true );

	/* Thumbs for popular widget */
	add_image_size( 'swt-popular', 80, 80, true );        	
	
	/* Shortcode blog post image size */
	add_image_size( 'swt-shortcode-blog', 420, 188, true ); 

	/* Shortcode Portfolio */
	add_image_size( 'swt-shortcode-portfolio', 480, 385, true );   
			
	/* Blog post image size */
	add_image_size( 'swt-big', 690, 310, true );        	
	
}

/**
 * Registers nav menu locations.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function swt_register_menus() {
	register_nav_menu( 'primary',    _x( 'Primary',    'nav menu location', 'swt' ) );
	register_nav_menu( 'social',     _x( 'Social',    'nav menu location', 'swt' ) );
}

/**
 * Registers sidebars.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function swt_register_sidebars() {
 
	hybrid_register_sidebar(
		array(
			'id'          => 'primary',
			'name'        => _x( 'Primary', 'sidebar', 'swt' ),
			'description' => __( 'The main sidebar. It is displayed on either the left or right side of the page based on the chosen layout.', 'swt' )
		)
	);
			
	hybrid_register_sidebar(
		array(
			'id'          => 'footer1',
			'name'        => _x( 'Footer - column 1', 'sidebar', 'swt' )			
		)
	);	
	
	global $col_number; 
		
	if ( $col_number > 1 ) {
	
		hybrid_register_sidebar(
			array(
				'id'          => 'footer2',
				'name'        => _x( 'Footer - column 2', 'sidebar', 'swt' )			
			)
		);	
	
	} 

	if ( $col_number == 3 ) {
	
		hybrid_register_sidebar(
			array(
				'id'          => 'footer3',
				'name'        => _x( 'Footer - column 3', 'sidebar', 'swt' )			
			)
		);	
	}
	
}

 
/**
 * Enqueues scripts.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function swt_enqueue_scripts() {

	wp_register_script( 'swt-responsive-nav', trailingslashit( get_template_directory_uri() ) . 'js/responsive-nav.min.js', array( 'jquery' ), null, true );
	wp_enqueue_script( 'swt-responsive-nav' );

	wp_register_script( 'swt-fitvids', trailingslashit( get_template_directory_uri() ) . 'js/fitvids.js', array( 'jquery' ), null, true );
	wp_enqueue_script( 'swt-fitvids' );
	
	wp_register_script( 'swt-custom-js', trailingslashit( get_template_directory_uri() ) . 'js/custom.js', array( 'jquery' ), null, true );
	wp_enqueue_script( 'swt-custom-js' );
  	
	wp_register_script( 'swt-slider', trailingslashit( get_template_directory_uri() ) . 'js/slider.js', array( 'jquery' ), null, true );
	
	wp_register_script( 'swt-wow-js', trailingslashit( get_template_directory_uri() ) . 'js/wow.min.js', array( 'jquery' ), null, true );	
	
	if ( function_exists( 'wpcf7' ) ) 
		wp_enqueue_script( 'swt-autosize', trailingslashit( get_template_directory_uri() ) . 'js/jquery.autosize.min.js', array( 'jquery' ), null, true );
	
}

/**
 * Registers custom stylesheets for the front end.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function swt_register_styles() {
	
	/* Font Icon Font */
	wp_register_style( 'font-awesome', '//netdna.bootstrapcdn.com/font-awesome/4.2.0/css/font-awesome.css' );	
	wp_enqueue_style( 'font-awesome' );
	
	/* Animate.css */
	wp_register_style( 'swt-animate-css', trailingslashit( get_template_directory_uri() ) .'css/animate.css' );		
	
	/* Theme Fonts */
	wp_register_style( 'font-merriweather', 'http://fonts.googleapis.com/css?family=Merriweather:700italic,700,400&subset=latin,latin-ext' );	
	wp_enqueue_style( 'font-merriweather' );	
	wp_register_style( 'font-open-sans', 'http://fonts.googleapis.com/css?family=Open+Sans:400,700&subset=latin,latin-ext' );	
	wp_enqueue_style( 'font-open-sans' );		
	
	/* Remove "Like Dislike Counter" default CSS */
	remove_action( 'wp_print_styles', 'ldc_lite_enqueue_css' );
			
}
 

/**
 * Adds a custom excerpt length.
 *
 * @since  1.0.0
 * @access public
 * @param  int     $length
 * @return int
 */
function swt_excerpt_length( $length ) {
	return 60;
}

function swt_read_more( $more ) {
	global $post;
	return '';	
}

/* Adds class to subsidiary sidebar (one-third, one-half or full-width-sidebar) */
function footer_column_number( $attr, $context ) {
	
	if ( 'subsidiary' === $context ) {
		
		global $col_number; 
				
		if ( $col_number == 3 ) 
			$attr['class']  = 'one-third';
		elseif( $col_number == 2 )
			$attr['class']  = 'one-half';
		else 
			$attr['class']  = 'full-width-sidebar';

	}
	
	return $attr;
	
}

/* Ads .wrap class */
function loop_meta_class( $attr, $context ) {
	$attr['class'] = 'loop-meta wrap';
	return $attr;
}

/*
* Default layout
*/
function main_theme_layout( $layout ) {

	if ( !is_singular() ) {
		$layout = swt_get_option( 'site_layout', 'layout-2c-l' );
	}
	elseif ( is_single() ) {
		if ( get_post_layout( get_the_ID() ) == 'default' ){
		        $layout = swt_get_option( 'site_layout', 'layout-2c-l' );
		}
	}
	elseif ( is_page() ) {
		if ( get_post_layout( get_the_ID() ) == 'default' ){
		       $layout = swt_get_option( 'site_layout', 'layout-2c-l' );
		}
	} 
	else {
		$layout = 'layout-'.get_post_layout( get_the_ID() );
	}    	

	return $layout;
}

/**
 * Disables sidebars if viewing a one-column page.
 *
 * @since 0.1.0
 */
function swt_disable_sidebars( $sidebars_widgets ) {
	
	global $wp_query, $col_number;

	if ( current_theme_supports( 'theme-layouts' ) ) {

		if ( 'layout-1c' == theme_layouts_get_layout() ) {
			$sidebars_widgets['primary'] = false;
		}
                
	}
	
	if ( is_page_template( 'templates/sections.php' ) )
		$sidebars_widgets['primary'] = false;
	
	/* Disable sidebar widgets in footer if "Number of columns in footer..." is 1 or 2  */
	if ( $col_number == 1 ) {
		$sidebars_widgets['footer2'] = false;	
		$sidebars_widgets['footer3'] = false;
	}
	elseif ( $col_number == 2 ) {
		$sidebars_widgets['footer3'] = false;			
	}
		

	return $sidebars_widgets;
}

/**
 * Function for deciding which pages should have a one-column layout.
 *
 * @since 0.1.0
 */ 
function swt_one_column() {

	if ( !is_active_sidebar( 'primary' ) || is_attachment() && wp_attachment_is_image() )
		add_filter( 'get_theme_layout', 'swt_theme_layout_one_column' );

	elseif ( is_attachment() && 'layout-default' == theme_layouts_get_layout() )
		add_filter( 'get_theme_layout', 'swt_theme_layout_one_column' );
}

/**
 * Filters 'get_theme_layout' by returning 'layout-1c'.
 *
 * @since 0.2.0
 */
function swt_theme_layout_one_column( $layout ) {
	return 'layout-1c';
}

/* Modifies default comment form */
function swt_modify_commentform( $defaults ) {
 
	$commenter = wp_get_current_commenter();
	$req = get_option( 'require_name_email' );
	$aria_req = ( $req ? " aria-required='true'" : '' );	
	
	$defaults['fields'] = array( 
		
		'author' =>
			'<p class="comment-form-author">' .						
			'<input id="author" placeholder="'. __( 'Name', 'swt' ) .'" name="author" type="text" value="' . esc_attr( $commenter['comment_author'] ) .
			'" size="30"' . $aria_req . ' /></p>',

		'email' =>
			'<p class="comment-form-email">' .
			'<input id="email" placeholder="'. __( 'Email', 'swt' ) .'" name="email" type="text" value="' . esc_attr(  $commenter['comment_author_email'] ) .
			'" size="10"' . $aria_req . ' /></p>',

		'url' =>
			'<p class="comment-form-url">' .
			'<input id="url" name="url" placeholder="'. __( 'Website', 'swt' ) .'" type="text" value="' . esc_attr( $commenter['comment_author_url'] ) .
			'" size="30" /></p>'
			
	);
		
		
	$defaults['comment_field'] = '<p class="comment-form-comment">'.
								 '<textarea placeholder="'. __( 'Your message', 'swt' ) .'" id="comment" name="comment" rows="8" cols="45"'. 
								 'aria-required="true"></textarea></p>';
								 
	$defaults['comment_notes_after'] = '';
	return $defaults;
	 
}

/* Functions from theme options - header scripts, custom css and favicon */
function swt_head_hooks() {
	
	$header_scripts = swt_get_option( 'header_scripts' ); 
	$custom_css = swt_get_option( 'custom_css' );
	$favicon = swt_get_option( 'favicon' ); 
	$output = '';
	
	if ( $favicon ) { 
		$output .= "<link rel=\"shortcut icon\" href=\"{$favicon}\" />\n";
	}		
	
	if ( $header_scripts ) 
		$output .= $header_scripts ."\n";
	
	if ( $custom_css ) 
		$output .= "<style type=\"text/css\">\n". $custom_css ."\n</style>\n";
	
	echo $output; 
}

/* Adds footer scripts */
function swt_footer_scripts() {
	
	$footer_scripts = swt_get_option( 'footer_scripts' ); 
	
	if ( $footer_scripts )
		echo $footer_scripts . "\n";

}

/*this function allows for the auto-creation of post excerpts*/
function truncate_post( $amount,$quote_after=false ) {
	$truncate = get_the_content();
	$truncate = apply_filters('the_content', $truncate);
	$truncate = preg_replace('@<script[^>]*?>.*?</script>@si', '', $truncate);
	$truncate = preg_replace('@<style[^>]*?>.*?</style>@si', '', $truncate);
	$truncate = strip_tags($truncate);
	$truncate = substr($truncate, 0, strrpos(substr($truncate, 0, $amount), ' '));
	echo $truncate;
	echo "...";
	if ($quote_after) echo('');
} 

/*
** Top header button
*/

function swt_header_button() {
	
	if ( swt_get_option( 'swt_header_button', 0 ) == true ) {
		
		$text = swt_get_option( 'swt_header_button_text' );
		$link = swt_get_option( 'swt_header_button_link' );
		
		if ( empty( $text ) ) {
			return;
		} 
		elseif ( empty( $link ) ) {
			echo "<span class=\"header-button\">{$text}</span>";
		} else {
			echo "<a class=\"header-button\" href=\"{$link}\"><span>{$text}</span></a>";
		}
					
	} 
}

/*
** Adds span around site title letters
*/
function add_span_2_site_title( $title ) {

	if ( swt_get_option( 'site_title_span', 1 ) == 1 ) {
	
		if ( $title = get_bloginfo( 'name' ) ) {		
						
			$title = str_split( $title, swt_get_option( 'site_title_span_letters_num', 1 ) ); 
			$title[0] = "<span>{$title[0]}</span>";
			$title = join( $title );
			
			$title = sprintf( '<h1 %s><a href="%s" rel="home">%s</a></h1>', hybrid_get_attr( 'site-title' ), home_url(), $title );
		}
		
	}
	
	return $title;
}

/*
** Ads extra classes depending on turned on options
*/
function extra_body_classes( $classes ) {

	$social_menu 	= has_nav_menu( 'social' );
	$header_button  = swt_get_option( 'swt_header_button', 0 );
	
	if ( $social_menu && $header_button == 1 ) 
		$classes[] = 'menu-and-button-on';
 	
	return $classes;
}

/* 
* Pulls out first gallery from gallery post format and puts images into slider. 
* since 1.0
*/
function strip_first_gallery( $content ) {
	wp_enqueue_script( 'swt-slider' );
	if ( 'gallery' === get_post_format() ) {
 
		preg_match('/\[gallery.*ids=.(.*).\]/', $content, $match_gallery); 		
		
		if ( !empty( $match_gallery ) )			
			$content = str_replace( $match_gallery[0], "", $content );
		
	}
	return $content;
 
}

/** 
* Function will return array of image/attachment data (alt, caption, src, title...)
*
* @since 1.0
*/
function swt_get_attachment_array( $id ) {

	$attachment = get_post( $id );
	
	return array(
		'alt' 			=> get_post_meta( $attachment->ID, '_wp_attachment_image_alt', true ),
		'caption' 		=> $attachment->post_excerpt,
		'href' 			=> get_permalink( $attachment->ID ),
		'src' 			=> $attachment->guid,
		'title'		 	=> $attachment->post_title,
		'description' 	=> $attachment->post_content		
	);
}

/*
* Function returns image id from URL 
*
* @since 1.0
*/
function swt_get_attachment_id_from_url( $attachment_url = '' ) {
 
	global $wpdb;
	$attachment_id = false;
 
	// If there is no url, return.
	if ( '' == $attachment_url )
		return;
 
	// Get the upload directory paths
	$upload_dir_paths = wp_upload_dir();
 
	// Make sure the upload path base directory exists in the attachment URL, to verify that we're working with a media library image
	if ( false !== strpos( $attachment_url, $upload_dir_paths['baseurl'] ) ) {
 
		// If this is the URL of an auto-generated thumbnail, get the URL of the original image
		$attachment_url = preg_replace( '/-\d+x\d+(?=\.(jpg|jpeg|png|gif)$)/i', '', $attachment_url );
 
		// Remove the upload path base directory from the attachment URL
		$attachment_url = str_replace( $upload_dir_paths['baseurl'] . '/', '', $attachment_url );
 
		// Finally, run a custom database query to get the attachment ID from the modified attachment URL
		$attachment_id = $wpdb->get_var( $wpdb->prepare( "SELECT wposts.ID FROM $wpdb->posts wposts, $wpdb->postmeta wpostmeta WHERE wposts.ID = wpostmeta.post_id AND wpostmeta.meta_key = '_wp_attached_file' AND wpostmeta.meta_value = '%s' AND wposts.post_type = 'attachment'", $attachment_url ) );
 
	}
 
	return $attachment_id;
}
