<?php
/**
 * Sets up custom filters and actions for the theme.  This does things like sets up sidebars, menus, scripts,
 * and lots of other awesome stuff that WordPress themes do.
 */

/* Register custom image sizes. */
add_action( 'init', 'swt_register_image_sizes', 5 );

/* Register sidebars. */
add_action( 'widgets_init', 'swt_register_sidebars', 5 );

/* Add custom scripts. */
add_action( 'wp_enqueue_scripts', 'swt_enqueue_scripts' );

/* Register custom styles. */
add_action( 'wp_enqueue_scripts',    'swt_register_styles', 0 );

/* Filters the excerpt length. */
add_filter( 'excerpt_length', 'swt_excerpt_length' );

/* Remove read more in excerpt */
add_filter( 'excerpt_more', 'swt_read_more' );

/* Remove allowed tags from comment form */
add_filter( 'comment_form_defaults', 'remove_comment_form_allowed_tags' );

/* Add header scripts, custom css and favicon to <head> */
add_action( 'wp_head', 'swt_head_hooks' );

/* Add footer scripts to </body> */
add_action( 'wp_footer', 'swt_footer_scripts' );

/* Allow more tags in Options Framework */
add_action( 'admin_init', 'optionscheck_change_santiziation', 100 );


/**
 * Registers custom image sizes for the theme.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function swt_register_image_sizes() {

	/* Sets the 'post-thumbnail' size. */
	set_post_thumbnail_size( 150, 150, true );

	/* Portfolio image fp */
	add_image_size( 'swt-portfolio-fp', 600, 450, true );

	/* Portfolio widget */
	add_image_size( 'swt-portfolio-widget', 300, 300, true );

	/* Full image */
	add_image_size( 'swt-full', 960 );

}

/**
 * Registers sidebars.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function swt_register_sidebars() {

	hybrid_register_sidebar(
		array(
			'id'          => 'primary',
			'name'        => _x( 'Primary', 'sidebar', 'swt' ),
			'description' => __( 'The main sidebar. It is displayed on either the left or right side of the page based on the chosen layout.', 'swt' )
		)
	);
}


/**
 * Enqueues scripts.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function swt_enqueue_scripts() {

	wp_enqueue_script( 'swt-custom-js', trailingslashit( get_template_directory_uri() ) . 'js/custom.js', array( 'jquery' ), null, true );
	wp_enqueue_script( 'swt-fitvids', trailingslashit( get_template_directory_uri() ) . 'js/fitvids.js', array( 'jquery' ), null, true );
	wp_enqueue_script( 'swt-slimScroll', trailingslashit( get_template_directory_uri() ) . 'js/slimscroll.js', array( 'jquery' ), null, true );

	if ( is_page_template( 'templates/template-portfolio.php' ) )
		wp_enqueue_script( 'swt-shuffle-js', trailingslashit( get_template_directory_uri() ) . 'js/jquery.shuffle.min.js', array( 'jquery' ), null, true );

}

/* Prints out data-groups for articles (needed for shuffle.js) */
function portfolio_data_groups() {

	if ( !is_singular() )
		return;

	global $post;

	$terms = get_the_terms( $post->ID, 'portfolio_category' );

	if ( $terms && ! is_wp_error( $terms ) ) {

		$data_group_items = array();

		foreach ( $terms as $term ) {
			$data_group_items[] = '"'.$term->name.'"';
		}

		$group_item = join( ", ", $data_group_items );
		echo " data-groups='[{$group_item}]'";
	}
}
/**
 * Registers custom stylesheets for the front end.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function swt_register_styles() {

	/* Theme Font */
	wp_register_style( 'pt-serif', 'http://fonts.googleapis.com/css?family=PT+Serif:400,700&subset=latin,latin-ext,cyrillic,cyrillic-ext' );
	wp_enqueue_style( 'pt-serif' );
}


/**
 * Adds a custom excerpt length.
 *
 * @since  1.0.0
 * @access public
 * @param  int     $length
 * @return int
 */
function swt_excerpt_length( $length ) {
	return 60;
}

function swt_read_more( $more ) {
	global $post;
	return '';
}

/* Disables comments allowed tags */
function remove_comment_form_allowed_tags( $defaults ) {

	$defaults['title_reply'] = '';
	$defaults['label_submit'] = __( 'Submit Comment', 'swt' );
	$defaults['comment_notes_before'] = '';
	$defaults['comment_notes_after'] = '';
    $defaults['comment_field'] = '<p class="comment-form-comment"><textarea id="comment" name="comment" cols="45" rows="8" aria-required="true"></textarea></p>';

	return $defaults;

}

/* Functions from theme options - header scripts, custom css and favicon */
function swt_head_hooks() {

	$header_scripts = of_get_option( 'header_scripts' );
	$custom_css = of_get_option( 'custom_css' );
	$favicon = of_get_option( 'favicon' );

	if ( $favicon ) {
		echo "<link rel=\"icon\" href=\"{$favicon}\" type=\"image/x-icon\" />\n";
	}

	if ( $header_scripts )
		echo $header_scripts ."\n";

	if ( $custom_css )
		echo "<style type=\"text/css\">\n". $custom_css ."\n</style>\n";
}

/* Adds footer scripts */
function swt_footer_scripts() {

	$footer_scripts = of_get_option( 'footer_scripts' );

	if ( $footer_scripts )
		echo $footer_scripts . "\n";

}

/*this function allows for the auto-creation of post excerpts*/
function truncate_post( $amount,$quote_after=false ) {
	$truncate = get_the_content();
	$truncate = apply_filters('the_content', $truncate);
	$truncate = preg_replace('@<script[^>]*?>.*?</script>@si', '', $truncate);
	$truncate = preg_replace('@<style[^>]*?>.*?</style>@si', '', $truncate);
	$truncate = strip_tags($truncate);
	$truncate = substr($truncate, 0, strrpos(substr($truncate, 0, $amount), ' '));
	echo $truncate;
	echo "...";
	if ($quote_after) echo('');
}

/* Change sanitize options for Options Framework */
function optionscheck_change_santiziation() {
    remove_filter( 'of_sanitize_textarea', 'of_sanitize_textarea' );
    add_filter( 'of_sanitize_textarea', 'custom_sanitize_textarea' );
}

function custom_sanitize_textarea( $input ) {

    global $allowedposttags;

	$custom_allowedtags["script"] =
		array(
			"text" => array(),
			"type" => array()
		);

	$custom_allowedtags = array_merge( $custom_allowedtags, $allowedposttags );
	$output = wp_kses( $input, $custom_allowedtags );

	return $output;
}
