<?php
/**
 * Sets up custom filters and actions for the theme.  This does things like sets up sidebars, menus, scripts, 
 * and lots of other awesome stuff that WordPress themes do.
 */

/* Register custom image sizes. */
add_action( 'init', 'swt_register_image_sizes', 5 );

/* Register custom menus. */
add_action( 'init', 'swt_register_menus', 5 );

/* Register sidebars. */
add_action( 'widgets_init', 'swt_register_sidebars', 5 );

/* Add custom scripts. */
add_action( 'wp_enqueue_scripts', 'swt_enqueue_scripts' );

/* Register custom styles. */
add_action( 'wp_enqueue_scripts',    'swt_register_styles', 0 );

/* Exclude sticky posts from home page. */
add_action( 'pre_get_posts', 'swt_exclude_sticky' );

/* Filters the excerpt length. */
add_filter( 'excerpt_length', 'swt_excerpt_length' );

/* Remove read more in excerpt */
add_filter( 'excerpt_more', 'swt_read_more' );

/* Main theme layout */
add_filter( 'get_theme_layout', 'main_theme_layout' );

/* Disable sidebar if 1 column layout */
add_filter( 'sidebars_widgets', 'swt_disable_sidebars' );
add_action( 'template_redirect', 'swt_one_column' );	

/* Remove allowed tags from comment form */
add_filter( 'comment_form_defaults', 'remove_comment_form_allowed_tags' );

/* Add header scripts, custom css and favicon to <head> */
add_action( 'wp_head', 'swt_head_hooks' );

/* Add footer scripts to </body> */
add_action( 'wp_footer', 'swt_footer_scripts' ); 

/* Allow more tags in Options Framework */
add_action( 'admin_init', 'optionscheck_change_santiziation', 100 );

 
 
/**
 * Registers custom image sizes for the theme.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function swt_register_image_sizes() {

	/* Sets the 'post-thumbnail' size. */
	set_post_thumbnail_size( 110, 110, true );
	
	/* Slider image size */
	add_image_size( 'swt-slider-image', 1000, 440, true );   

	/* Featured widget image */
	add_image_size( 'swt-featured-image', 280, 280, true );        
	
}

/**
 * Registers nav menu locations.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function swt_register_menus() {
	register_nav_menu( 'primary',    _x( 'Primary',    'nav menu location', 'swt' ) );
}

/**
 * Registers sidebars.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function swt_register_sidebars() {
 
	hybrid_register_sidebar(
		array(
			'id'          => 'header',
			'name'        => _x( 'Header Sidebar', 'sidebar', 'swt' ),
			'description' => __( 'A sidebar in top right corner of the header, suitable for banner, search form or whatever it\'s useful to you.', 'swt' )
		)
	); 
 
	hybrid_register_sidebar(
		array(
			'id'          => 'primary',
			'name'        => _x( 'Primary', 'sidebar', 'swt' ),
			'description' => __( 'The main sidebar. It is displayed on either the left or right side of the page based on the chosen layout.', 'swt' )
		)
	);

	hybrid_register_sidebar(
		array(
			'id'          => 'subsidiary',
			'name'        => _x( 'Subsidiary', 'sidebar', 'swt' ),
			'description' => __( 'A sidebar located in the footer of the site. Optimized for one, two, three or four widgets (and multiples thereof).', 'swt' )
		)
	);
}

/**
 * Excluding sticky posts from home page if slider enabled. Sticky posts are in a slider.
 * 
 * @since 0.1
 */
function swt_exclude_sticky( $query ) {
	
	/* Exclude if is home and is main query. */
	if ( is_home() && $query->is_main_query() && of_get_option( 'swt_slider' ) == 1 )
		$query->set( 'post__not_in', get_option( 'sticky_posts' ) );
	
}

/**
 * Enqueues scripts.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function swt_enqueue_scripts() {

	wp_enqueue_script( 'swt-custom-js', trailingslashit( get_template_directory_uri() ) . 'js/custom.js', array( 'jquery' ), null, true );
	wp_enqueue_script( 'swt-fitvids', trailingslashit( get_template_directory_uri() ) . 'js/fitvids.js', array( 'jquery' ), null, true );
	wp_enqueue_script( 'swt-responsive-nav', trailingslashit( get_template_directory_uri() ) . 'js/responsive-nav.min.js' );	
	
	global $sticky, $sticky_slider;
	
	$sticky = get_option( 'sticky_posts' );
    $sticky_slider = of_get_option( 'swt_slider' );	
	
	/* Enqueue slider.js if slider activated and blog has sticky posts. */
	if ( $sticky_slider == 1 && ! empty( $sticky ) && ( is_home() || is_front_page() ) ) {
	
			wp_enqueue_script( 'swt-slider', trailingslashit( get_template_directory_uri() ) . 'js/slider.js', array( 'jquery' ), null, true );
	
			$slider_mode = of_get_option( 'slider_mode', 'fade' ); 
			$slider_auto = of_get_option( 'slider_auto', 'true' ); 
			$slider_pause = of_get_option( 'slider_pause', 4000 );
	
			$params = array( 'mode' => $slider_mode,
							'auto' => $slider_auto,
							'pause' => $slider_pause
			);	
    
			wp_localize_script( 'swt-custom-js', 'Params', $params ); 
	
	}
	
	/* Infinite Scroll */
	wp_register_script( 'infinite_scroll',  
						trailingslashit( get_template_directory_uri() ). 'js/infinitescroll.min.js', array( 'jquery' ), null, true );
	wp_register_script( 'manual_trigger', trailingslashit( get_template_directory_uri() ) . 'js/manual-trigger.js', array( 'infinite_scroll' ), null, true );	
	
	if( ! is_singular() ) {
		
		wp_enqueue_script( 'infinite_scroll' );
		wp_enqueue_script( 'manual_trigger' );
	
		add_action( 'wp_footer', 'custom_infinite_scroll_js', 100 );
		add_action( 'wp', 'custom_paged_404_fix' );		
		
	}	
			
}

/* Add infinite scroll script setup to the footer */
function custom_infinite_scroll_js() {
	{ ?>
		<script>
		jQuery('.post-wrapper').infinitescroll({
			navSelector  : "#ajax_load_posts",
			nextSelector : "#ajax_load_posts a",
			itemSelector : ".entry",
			behavior: "twitter",
			loading: {
				img: '<?php echo trailingslashit( get_template_directory_uri() ) .'images/infinite-preloader.gif'; ?>',
				msgText: "",
				finishedMsg: "<em><?php _e( 'All posts loaded!', 'swt' ); ?></em>"
 			}
		});	
		
		</script>
	<?php
	}
}

function custom_paged_404_fix( ) {
	global $wp_query;

	if ( is_404() || !is_paged() || 0 != count( $wp_query->posts ) )
		return;

	$wp_query->set_404();
	status_header( 404 );
	nocache_headers();
}

/**
 * Registers custom stylesheets for the front end.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function swt_register_styles() {
	
	/* Font Icon Font */
	wp_register_style( 'font-awesome', '//netdna.bootstrapcdn.com/font-awesome/4.1.0/css/font-awesome.css' );	
	wp_enqueue_style( 'font-awesome' );
	
}
 

/**
 * Adds a custom excerpt length.
 *
 * @since  1.0.0
 * @access public
 * @param  int     $length
 * @return int
 */
function swt_excerpt_length( $length ) {
	return 60;
}

function swt_read_more( $more ) {
	global $post;
	return '';		
}

/*
* Default layout
*/
function main_theme_layout( $layout ) {

	if ( !is_singular() ) {
		$layout = of_get_option( 'site_layout', 'layout-2c-l' );
	}
	elseif ( is_single() ) {
		if ( get_post_layout( get_the_ID() ) == 'default' ){
		        $layout = of_get_option( 'site_layout', 'layout-2c-l' );
		}
	}
	elseif ( is_page() ) {
		if ( get_post_layout( get_the_ID() ) == 'default' ){
		        $layout = of_get_option( 'site_layout', 'layout-2c-l' );
		}
	} 
	else {
		$layout = 'layout-'.get_post_layout( get_the_ID() );
	}    	

	return $layout;
}


/**
 * Disables sidebars if viewing a one-column page.
 *
 * @since 0.1.0
 */
function swt_disable_sidebars( $sidebars_widgets ) {
	
	global $wp_query;

	if ( current_theme_supports( 'theme-layouts' ) ) {

		if ( 'layout-1c' == theme_layouts_get_layout() ) {
			$sidebars_widgets['primary'] = false;
		}
                
	}

	return $sidebars_widgets;
}

/**
 * Function for deciding which pages should have a one-column layout.
 *
 * @since 0.1.0
 */ 
function swt_one_column() {

	if ( !is_active_sidebar( 'primary' ) || is_attachment() && wp_attachment_is_image() )
		add_filter( 'get_theme_layout', 'swt_theme_layout_one_column' );

	elseif ( is_attachment() && 'layout-default' == theme_layouts_get_layout() )
		add_filter( 'get_theme_layout', 'swt_theme_layout_one_column' );
}

/**
 * Filters 'get_theme_layout' by returning 'layout-1c'.
 *
 * @since 0.2.0
 */
function swt_theme_layout_one_column( $layout ) {
	return 'layout-1c';
}

/* Disables comments allowed tags */
function remove_comment_form_allowed_tags( $defaults ) {
 
	$defaults['comment_notes_after'] = '';
	return $defaults;
	 
}

/* Functions from theme options - header scripts, custom css and favicon */
function swt_head_hooks() {
	
	$header_scripts = of_get_option( 'header_scripts' ); 
	$custom_css = of_get_option( 'custom_css' );
	$favicon = of_get_option( 'favicon' ); 
	
	if ( $favicon ) { 
		echo "<link rel=\"icon\" href=\"{$favicon}\" type=\"image/x-icon\" />\n";
	}		
	
	if ( $header_scripts ) 
		echo $header_scripts ."\n";
	
	if ( $custom_css ) 
		echo "<style type=\"text/css\">\n". $custom_css ."\n</style>\n";
}

/* Adds footer scripts */
function swt_footer_scripts() {
	
	$footer_scripts = of_get_option( 'footer_scripts' ); 
	
	if ( $footer_scripts )
		echo $footer_scripts . "\n";

}

/*this function allows for the auto-creation of post excerpts*/
function truncate_post( $amount,$quote_after=false ) {
	$truncate = get_the_content();
	$truncate = apply_filters('the_content', $truncate);
	$truncate = preg_replace('@<script[^>]*?>.*?</script>@si', '', $truncate);
	$truncate = preg_replace('@<style[^>]*?>.*?</style>@si', '', $truncate);
	$truncate = strip_tags($truncate);
	$truncate = substr($truncate, 0, strrpos(substr($truncate, 0, $amount), ' '));
	echo $truncate;
	echo "...";
	if ($quote_after) echo('');
} 

/* Change sanitize options for Options Framework */
function optionscheck_change_santiziation() {
    remove_filter( 'of_sanitize_textarea', 'of_sanitize_textarea' );
    add_filter( 'of_sanitize_textarea', 'custom_sanitize_textarea' );
}

function custom_sanitize_textarea( $input ) {

    global $allowedposttags;
	
	$custom_allowedtags["script"] = 
		array(
			"text" => array(),
			"type" => array()
		);
	
	$custom_allowedtags = array_merge( $custom_allowedtags, $allowedposttags );
	$output = wp_kses( $input, $custom_allowedtags );
	
	return $output;
}